<?php
/**
 * PAYHIIVE API Handler
 *
 * Handles all API communication with PAYHIIVE.
 *
 * @package PayHive_Payments
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * PayHive_API class.
 */
class PayHive_API
{

    /**
     * API Base URL for sandbox mode.
     *
     * @var string
     */
    private $sandbox_url = 'https://payhiive.com/api/v1';

    /**
     * API Base URL for live mode.
     *
     * @var string
     */
    private $live_url = 'https://payhiive.com/api/v1';

    /**
     * Get the base URL based on current mode.
     *
     * @return string
     */
    private function get_base_url()
    {
        $mode = get_option('payhive_mode', 'sandbox');
        return ('live' === $mode) ? $this->live_url : $this->sandbox_url;
    }

    /**
     * Get API credentials.
     *
     * @return array
     */
    private function get_credentials()
    {
        return array(
            'api_key' => get_option('payhive_api_key', ''),
            'secret_key' => get_option('payhive_secret_key', ''),
        );
    }

    /**
     * Log API request and response.
     *
     * @param string $endpoint The API endpoint.
     * @param array  $request  The request data.
     * @param mixed  $response The response data.
     * @param bool   $is_error Whether this is an error log.
     */
    private function log($endpoint, $request, $response, $is_error = false)
    {
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            return;
        }

        $mode = get_option('payhive_mode', 'sandbox');
        $log_type = $is_error ? 'ERROR' : 'INFO';

        $log_message = sprintf(
            "[PayHive %s] [%s] Endpoint: %s\nRequest: %s\nResponse: %s",
            $log_type,
            strtoupper($mode),
            $endpoint,
            wp_json_encode($request, JSON_PRETTY_PRINT),
            is_wp_error($response) ? $response->get_error_message() : wp_json_encode($response, JSON_PRETTY_PRINT)
        );

        error_log($log_message);
    }

    /**
     * Make API request.
     *
     * @param string $endpoint    The API endpoint (e.g., '/payment-links').
     * @param array  $data        The request data.
     * @param string $method      The HTTP method (GET, POST, etc.).
     * @return array|WP_Error
     */
    private function make_request($endpoint, $data = array(), $method = 'POST')
    {
        $credentials = $this->get_credentials();

        // Validate credentials
        if (empty($credentials['api_key']) || empty($credentials['secret_key'])) {
            $error = new WP_Error(
                'payhive_missing_credentials',
                __('PayHive API credentials are not configured. Please check your settings.', 'payhive-payments')
            );
            $this->log($endpoint, $data, $error, true);
            return $error;
        }

        $url = $this->get_base_url() . $endpoint;
        
        // For GET requests, add query parameters if data is provided
        if ('GET' === $method && !empty($data) && is_array($data)) {
            $query_string = http_build_query($data);
            $url .= (strpos($url, '?') !== false ? '&' : '?') . $query_string;
        }

        $args = array(
            'method' => $method,
            'headers' => array(
                'X-Public-Key' => $credentials['api_key'],
                'X-Secret-Key' => $credentials['secret_key'],
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'timeout' => 30,
        );

        if ('POST' === $method || 'PUT' === $method) {
            $args['body'] = wp_json_encode($data);
        }

        // Make request
        $response = wp_remote_request($url, $args);

        // Log request and response
        $this->log($endpoint, $data, $response, is_wp_error($response));

        // Handle response
        if (is_wp_error($response)) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $decoded_response = json_decode($response_body, true);

        if ($response_code < 200 || $response_code >= 300) {
            // Try to get detailed error message - prioritize validation errors
            $error_message = __('An error occurred while communicating with PayHive API.', 'payhive-payments');
            
            // First check for Laravel validation errors (most specific)
            if (isset($decoded_response['errors']) && is_array($decoded_response['errors'])) {
                $errors = array();
                foreach ($decoded_response['errors'] as $field => $messages) {
                    if (is_array($messages)) {
                        $errors[] = ucfirst($field) . ': ' . implode(', ', $messages);
                    } else {
                        $errors[] = $messages;
                    }
                }
                if (!empty($errors)) {
                    $error_message = implode(' | ', $errors);
                }
            }
            // Then check for direct message
            elseif (isset($decoded_response['message']) && !empty($decoded_response['message'])) {
                $error_message = $decoded_response['message'];
            }
            // Then check for error field
            elseif (isset($decoded_response['error'])) {
                $error_message = is_array($decoded_response['error']) 
                    ? implode(', ', $decoded_response['error']) 
                    : $decoded_response['error'];
            }
            
            // Add error code if available and not already in message
            if (isset($decoded_response['error_code']) && strpos($error_message, $decoded_response['error_code']) === false) {
                $error_message .= ' (Code: ' . $decoded_response['error_code'] . ')';
            }

            $error = new WP_Error(
                'payhive_api_error',
                $error_message,
                array(
                    'status' => $response_code,
                    'response' => $decoded_response,
                )
            );

            $this->log($endpoint, $data, $error, true);
            return $error;
        }

        return $decoded_response;
    }

    /**
     * Create a payment link.
     *
     * @param array $payment_data Payment data.
     * @return array|WP_Error
     */
    public function create_payment_link($payment_data)
    {
        // Check if multiple amounts are enabled
        $enable_multiple_amounts = isset($payment_data['amounts']) && is_array($payment_data['amounts']) && !empty($payment_data['amounts']);
        
        if ($enable_multiple_amounts) {
            // Multiple amounts mode
            $data = array(
                'enable_multiple_amounts' => true,
                'amounts_json' => json_encode($payment_data['amounts']),
                'allow_custom_amount' => isset($payment_data['allow_custom_amount']) ? ($payment_data['allow_custom_amount'] ? '1' : '0') : '1',
                'currency' => isset($payment_data['currency']) ? $payment_data['currency'] : 'UGX',
                'description' => sanitize_text_field($payment_data['description']),
            );
        } else {
            // Single amount mode
            $data = array(
                'amount' => floatval($payment_data['amount']),
                'currency' => isset($payment_data['currency']) ? $payment_data['currency'] : 'UGX',
                'description' => sanitize_text_field($payment_data['description']),
            );
        }

        // Add optional fields
        if (isset($payment_data['expires_at']) && !empty($payment_data['expires_at'])) {
            $data['expires_at'] = sanitize_text_field($payment_data['expires_at']);
        }

        if (isset($payment_data['customer_email']) && !empty($payment_data['customer_email'])) {
            $data['customer_email'] = sanitize_email($payment_data['customer_email']);
        }

        if (isset($payment_data['customer_name']) && !empty($payment_data['customer_name'])) {
            $data['customer_name'] = sanitize_text_field($payment_data['customer_name']);
        }

        if (isset($payment_data['success_url']) && !empty($payment_data['success_url'])) {
            $data['success_url'] = esc_url_raw($payment_data['success_url']);
        }

        if (isset($payment_data['cancel_url']) && !empty($payment_data['cancel_url'])) {
            $data['cancel_url'] = esc_url_raw($payment_data['cancel_url']);
        }

        if (isset($payment_data['metadata']) && !empty($payment_data['metadata'])) {
            $data['metadata'] = sanitize_text_field($payment_data['metadata']);
        }

        // Use the API payment links endpoint
        return $this->make_request('/payment-links', $data, 'POST');
    }

    /**
     * Get payment links.
     *
     * @return array|WP_Error
     */
    public function get_payment_links()
    {
        return $this->make_request('/payment-links', array(), 'GET');
    }

    /**
     * Delete a payment link.
     *
     * @param int $link_id The payment link ID.
     * @return array|WP_Error
     */
    public function delete_payment_link($link_id)
    {
        $link_id = intval($link_id);
        return $this->make_request('/payment-links/' . $link_id, array(), 'DELETE');
    }

    /**
     * Get payment status.
     *
     * @param string $payment_id The payment ID.
     * @return array|WP_Error
     */
    public function get_payment_status($payment_id)
    {
        $payment_id = sanitize_text_field($payment_id);
        return $this->make_request('/payments/' . $payment_id, array(), 'GET');
    }

    /**
     * Process a mobile money payment.
     *
     * @param array $payment_data Payment data including phone number and provider.
     * @return array|WP_Error
     */
    public function process_mobile_payment($payment_data)
    {
        // Convert provider format (mtn -> MTN_MOMO_UGA, airtel -> AIRTEL_OAPI_UGA)
        $provider_map = array(
            'mtn' => 'MTN_MOMO_UGA',
            'airtel' => 'AIRTEL_OAPI_UGA',
        );
        
        $provider = strtolower(sanitize_text_field($payment_data['provider']));
        
        // If provider is already in API format (contains underscore), use as is
        if (strpos($provider, '_') !== false) {
            $api_provider = strtoupper($provider);
        } else {
            // Convert from simple format to API format
            $api_provider = isset($provider_map[$provider]) ? $provider_map[$provider] : 'MTN_MOMO_UGA';
        }
        
        // Validate provider format
        $valid_providers = array('MTN_MOMO_UGA', 'AIRTEL_OAPI_UGA');
        if (!in_array($api_provider, $valid_providers, true)) {
            return new WP_Error(
                'payhive_invalid_provider',
                __('Invalid mobile money provider. Must be MTN or Airtel.', 'payhive-payments')
            );
        }

        $data = array(
            'amount' => floatval($payment_data['amount']),
            'currency' => isset($payment_data['currency']) ? strtoupper($payment_data['currency']) : 'UGX',
            'phone_number' => sanitize_text_field($payment_data['phone']), // API expects phone_number
            'provider' => $api_provider, // API expects MTN_MOMO_UGA or AIRTEL_OAPI_UGA
        );
        
        // Add description if provided
        if (isset($payment_data['description']) && !empty($payment_data['description'])) {
            $data['description'] = sanitize_text_field($payment_data['description']);
        }

        // Add merchant_name if display merchant name is enabled
        $display_merchant_name = get_option('payhive_display_merchant_name', false);
        if ($display_merchant_name) {
            $merchant_name = get_option('payhive_merchant_name', 'Okao Emmanuel');
            if (!empty($merchant_name)) {
                $data['merchant_name'] = sanitize_text_field($merchant_name);
            }
        }

        // Log payment request for debugging
        $this->log(
            '/payments',
            array(
                'amount' => $data['amount'],
                'currency' => $data['currency'],
                'provider' => $data['provider'],
                'phone_number' => substr($data['phone_number'], -4) . '****', // Mask phone for logging
                'merchant_name' => isset($data['merchant_name']) ? $data['merchant_name'] : '(not included)',
                'description' => isset($data['description']) ? $data['description'] : '(not provided)',
            ),
            null,
            false
        );

        // Use POST /payments endpoint (not /payments/process)
        $response = $this->make_request('/payments', $data, 'POST');

        // Log response for debugging
        if (is_wp_error($response)) {
            $this->log('/payments', $data, $response, true);
        } else {
            $this->log('/payments', $data, $response, false);
        }

        return $response;
    }

    /**
     * Test API connection.
     *
     * @return bool|WP_Error
     */
    public function test_connection()
    {
        // Use a simpler health check endpoint if available, or just test authentication
        // Try GET /payments with minimal query parameters
        // Convert array to proper query string format
        $query_params = array(
            'per_page' => '1',
            'page' => '1'
        );
        
        // Build URL with query string manually to ensure proper format
        $base_url = $this->get_base_url();
        $endpoint = '/payments';
        $url = $base_url . $endpoint . '?' . http_build_query($query_params);
        
        $credentials = $this->get_credentials();
        
        // Validate credentials
        if (empty($credentials['api_key']) || empty($credentials['secret_key'])) {
            return new WP_Error(
                'payhive_missing_credentials',
                __('PayHive API credentials are not configured. Please check your settings.', 'payhive-payments')
            );
        }
        
        $args = array(
            'method' => 'GET',
            'headers' => array(
                'X-Public-Key' => $credentials['api_key'],
                'X-Secret-Key' => $credentials['secret_key'],
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ),
            'timeout' => 30,
        );
        
        // Make request
        $response = wp_remote_request($url, $args);
        
        // Log request and response
        $this->log($endpoint, $query_params, $response, is_wp_error($response));
        
        // Handle response
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $decoded_response = json_decode($response_body, true);
        
        if ($response_code < 200 || $response_code >= 300) {
            $error_message = __('An error occurred while communicating with PayHive API.', 'payhive-payments');
            
            if (isset($decoded_response['message']) && !empty($decoded_response['message'])) {
                $error_message = $decoded_response['message'];
            } elseif (isset($decoded_response['error'])) {
                $error_message = is_array($decoded_response['error']) 
                    ? implode(', ', $decoded_response['error']) 
                    : $decoded_response['error'];
            }
            
            $error = new WP_Error(
                'payhive_api_error',
                $error_message,
                array(
                    'status' => $response_code,
                    'response' => $decoded_response,
                )
            );
            
            $this->log($endpoint, $query_params, $error, true);
            return $error;
        }
        
        // Check if response indicates success
        if (isset($decoded_response['success']) && $decoded_response['success'] === true) {
            return true;
        }
        
        // If response doesn't have success field, assume connection worked
        // (some APIs might return data directly)
        return true;

        if (is_wp_error($response)) {
            return $response;
        }

        // Check if response indicates success
        // The GET /payments endpoint returns: { success: true, data: [...], pagination: {...} }
        if (isset($response['success']) && $response['success'] === true) {
            return true;
        }

        // If response doesn't have success field, assume connection worked
        // (some APIs might return data directly)
        return true;
    }
}
