<?php
/**
 * PAYHIIVE Payment Links Manager
 *
 * Handles payment link creation and management in WordPress admin.
 *
 * @package PayHive_Payments
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * PayHive_Payment_Links class.
 */
class PayHive_Payment_Links
{

    /**
     * The single instance of the class.
     *
     * @var PayHive_Payment_Links
     */
    protected static $_instance = null;

    /**
     * Main PayHive_Payment_Links Instance.
     *
     * @return PayHive_Payment_Links - Main instance.
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor.
     */
    public function __construct()
    {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('wp_ajax_payhive_create_payment_link', array($this, 'ajax_create_payment_link'));
        add_action('wp_ajax_payhive_get_payment_links', array($this, 'ajax_get_payment_links'));
        add_action('wp_ajax_payhive_delete_payment_link', array($this, 'ajax_delete_payment_link'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }

    /**
     * Add admin menu page.
     */
    public function add_admin_menu()
    {
        add_submenu_page(
            'payhive-payments',
            __('Payment Links', 'payhive-payments'),
            __('Payment Links', 'payhive-payments'),
            'manage_options',
            'payhive-payment-links',
            array($this, 'render_payment_links_page')
        );
    }

    /**
     * Enqueue admin scripts and styles.
     */
    public function enqueue_scripts($hook)
    {
        // Only load on our payment links page
        if ('settings_page_payhive-payment-links' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'payhive-admin-style',
            PAYHIVE_PLUGIN_URL . 'assets/css/admin-style.css',
            array(),
            PAYHIVE_VERSION
        );

        wp_enqueue_script(
            'payhive-payment-links-script',
            PAYHIVE_PLUGIN_URL . 'assets/js/payment-links.js',
            array('jquery'),
            PAYHIVE_VERSION,
            true
        );

        // Localize script
        wp_localize_script(
            'payhive-payment-links-script',
            'payhivePaymentLinks',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('payhive_payment_links_nonce'),
            )
        );
    }

    /**
     * AJAX handler for creating payment links.
     */
    public function ajax_create_payment_link()
    {
        // Verify nonce
        check_ajax_referer('payhive_payment_links_nonce', 'nonce');

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'payhive-payments')));
        }

        // Get and validate input
        $amount = isset($_POST['amount']) ? floatval($_POST['amount']) : 0;
        $description = isset($_POST['description']) ? sanitize_text_field($_POST['description']) : '';
        $currency = isset($_POST['currency']) ? sanitize_text_field($_POST['currency']) : 'UGX';
        $enable_multiple_amounts = isset($_POST['enable_multiple_amounts']) && $_POST['enable_multiple_amounts'] === '1';
        $amounts = isset($_POST['amounts']) && is_array($_POST['amounts']) ? array_map('floatval', $_POST['amounts']) : array();
        $allow_custom_amount = isset($_POST['allow_custom_amount']) && $_POST['allow_custom_amount'] === '1';
        $expires_at = isset($_POST['expires_at']) ? sanitize_text_field($_POST['expires_at']) : '';
        $customer_name = isset($_POST['customer_name']) ? sanitize_text_field($_POST['customer_name']) : '';
        $customer_email = isset($_POST['customer_email']) ? sanitize_email($_POST['customer_email']) : '';
        $success_url = isset($_POST['success_url']) ? esc_url_raw($_POST['success_url']) : '';
        $cancel_url = isset($_POST['cancel_url']) ? esc_url_raw($_POST['cancel_url']) : '';

        // Validation
        if (empty($description)) {
            wp_send_json_error(array('message' => __('Description is required.', 'payhive-payments')));
        }

        if ($enable_multiple_amounts) {
            if (empty($amounts)) {
                wp_send_json_error(array('message' => __('Please add at least one preset amount.', 'payhive-payments')));
            }
            foreach ($amounts as $amt) {
                if ($amt <= 0) {
                    wp_send_json_error(array('message' => __('All amounts must be valid positive numbers.', 'payhive-payments')));
                }
            }
        } else {
            if ($amount <= 0) {
                wp_send_json_error(array('message' => __('Amount must be greater than 0.', 'payhive-payments')));
            }
        }

        // Prepare payment data
        $payment_data = array(
            'description' => $description,
            'currency' => $currency,
        );

        if ($enable_multiple_amounts) {
            $payment_data['amounts'] = $amounts;
            $payment_data['allow_custom_amount'] = $allow_custom_amount;
            $payment_data['amount'] = min($amounts); // Use minimum as default
        } else {
            $payment_data['amount'] = $amount;
        }

        // Add optional fields
        if (!empty($customer_email)) {
            $payment_data['customer_email'] = $customer_email;
        }
        if (!empty($customer_name)) {
            $payment_data['customer_name'] = $customer_name;
        }
        if (!empty($success_url)) {
            $payment_data['success_url'] = $success_url;
        }
        if (!empty($cancel_url)) {
            $payment_data['cancel_url'] = $cancel_url;
        }
        if (!empty($expires_at)) {
            $payment_data['expires_at'] = $expires_at;
        }

        // Create payment link via API
        $api = new PayHive_API();
        $response = $api->create_payment_link($payment_data);

        if (is_wp_error($response)) {
            wp_send_json_error(array(
                'message' => $response->get_error_message(),
            ));
        }

        // Success
        $payment_link = '';
        $payment_id = '';
        
        if (isset($response['data'])) {
            $payment_link = isset($response['data']['payment_link']) ? $response['data']['payment_link'] : '';
            $payment_id = isset($response['data']['id']) ? $response['data']['id'] : (isset($response['data']['link_id']) ? $response['data']['link_id'] : '');
        } elseif (isset($response['payment_link'])) {
            $payment_link = $response['payment_link'];
            $payment_id = isset($response['id']) ? $response['id'] : '';
        }

        wp_send_json_success(array(
            'message' => __('Payment link created successfully!', 'payhive-payments'),
            'payment_link' => $payment_link,
            'payment_id' => $payment_id,
            'data' => $response,
        ));
    }

    /**
     * AJAX handler for getting payment links.
     */
    public function ajax_get_payment_links()
    {
        // Verify nonce
        check_ajax_referer('payhive_payment_links_nonce', 'nonce');

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'payhive-payments')));
        }

        // Get payment links via API
        $api = new PayHive_API();
        $response = $api->get_payment_links();

        if (is_wp_error($response)) {
            wp_send_json_error(array(
                'message' => $response->get_error_message(),
            ));
        }

        wp_send_json_success(array(
            'links' => isset($response['data']) ? $response['data'] : (isset($response['links']) ? $response['links'] : array()),
        ));
    }

    /**
     * AJAX handler for deleting payment links.
     */
    public function ajax_delete_payment_link()
    {
        // Verify nonce
        check_ajax_referer('payhive_payment_links_nonce', 'nonce');

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'payhive-payments')));
        }

        $link_id = isset($_POST['link_id']) ? intval($_POST['link_id']) : 0;

        if (empty($link_id)) {
            wp_send_json_error(array('message' => __('Link ID is required.', 'payhive-payments')));
        }

        // Delete payment link via API
        $api = new PayHive_API();
        $response = $api->delete_payment_link($link_id);

        if (is_wp_error($response)) {
            wp_send_json_error(array(
                'message' => $response->get_error_message(),
            ));
        }

        wp_send_json_success(array(
            'message' => __('Payment link deleted successfully!', 'payhive-payments'),
        ));
    }

    /**
     * Render payment links page.
     */
    public function render_payment_links_page()
    {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Payment Links', 'payhive-payments'); ?></h1>
            <p><?php esc_html_e('Create and manage payment links that you can share with your customers. Similar to the payment link creation in your PayHive dashboard.', 'payhive-payments'); ?></p>

            <div class="payhive-payment-links-container">
                <!-- Create Payment Link Form -->
                <div class="payhive-create-link-section">
                    <h2><?php esc_html_e('Create New Payment Link', 'payhive-payments'); ?></h2>
                    
                    <form id="payhive-create-link-form" class="payhive-form">
                        <table class="form-table">
                            <tr>
                                <th><label for="payhive_description"><?php esc_html_e('Description', 'payhive-payments'); ?> <span class="required">*</span></label></th>
                                <td>
                                    <input type="text" id="payhive_description" name="description" class="regular-text" required />
                                    <p class="description"><?php esc_html_e('A brief description of what this payment is for.', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="payhive_currency"><?php esc_html_e('Currency', 'payhive-payments'); ?></label></th>
                                <td>
                                    <select id="payhive_currency" name="currency">
                                        <option value="UGX" selected>UGX (Ugandan Shilling)</option>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th>
                                    <label>
                                        <input type="checkbox" id="payhive_enable_multiple_amounts" name="enable_multiple_amounts" value="1" />
                                        <?php esc_html_e('Enable Multiple Amounts', 'payhive-payments'); ?>
                                    </label>
                                </th>
                                <td>
                                    <p class="description"><?php esc_html_e('Allow users to select from preset amounts or enter a custom amount (useful for donations/charity).', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                            <tr id="payhive_single_amount_row">
                                <th><label for="payhive_amount"><?php esc_html_e('Amount', 'payhive-payments'); ?> <span class="required">*</span></label></th>
                                <td>
                                    <input type="number" id="payhive_amount" name="amount" class="small-text" step="0.01" min="0.01" />
                                    <span style="margin-left: 8px;">UGX</span>
                                    <p class="description"><?php esc_html_e('Payment amount in UGX.', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                            <tr id="payhive_multiple_amounts_row" style="display: none;">
                                <th><label><?php esc_html_e('Preset Amounts', 'payhive-payments'); ?> <span class="required">*</span></label></th>
                                <td>
                                    <div id="payhive_amounts_container">
                                        <div class="payhive-amount-input-group">
                                            <input type="number" name="amounts[]" class="small-text" step="0.01" min="0.01" placeholder="Amount" />
                                            <span style="margin-left: 8px;">UGX</span>
                                            <button type="button" class="button payhive-remove-amount" style="display: none;"><?php esc_html_e('Remove', 'payhive-payments'); ?></button>
                                        </div>
                                    </div>
                                    <button type="button" id="payhive_add_amount" class="button button-secondary" style="margin-top: 10px;">
                                        <?php esc_html_e('+ Add Amount', 'payhive-payments'); ?>
                                    </button>
                                    <p class="description"><?php esc_html_e('Add preset amounts that users can select from.', 'payhive-payments'); ?></p>
                                    <label style="margin-top: 10px; display: block;">
                                        <input type="checkbox" id="payhive_allow_custom_amount" name="allow_custom_amount" value="1" checked />
                                        <?php esc_html_e('Allow custom amount', 'payhive-payments'); ?>
                                    </label>
                                    <p class="description"><?php esc_html_e('Allow users to enter their own custom amount.', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="payhive_expires_at"><?php esc_html_e('Expires At', 'payhive-payments'); ?></label></th>
                                <td>
                                    <input type="datetime-local" id="payhive_expires_at" name="expires_at" />
                                    <p class="description"><?php esc_html_e('Optional: Set an expiration date for this payment link.', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="payhive_customer_name"><?php esc_html_e('Customer Name', 'payhive-payments'); ?></label></th>
                                <td>
                                    <input type="text" id="payhive_customer_name" name="customer_name" class="regular-text" />
                                    <p class="description"><?php esc_html_e('Optional: Pre-fill customer name.', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="payhive_customer_email"><?php esc_html_e('Customer Email', 'payhive-payments'); ?></label></th>
                                <td>
                                    <input type="email" id="payhive_customer_email" name="customer_email" class="regular-text" />
                                    <p class="description"><?php esc_html_e('Optional: Pre-fill customer email.', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="payhive_success_url"><?php esc_html_e('Success URL', 'payhive-payments'); ?></label></th>
                                <td>
                                    <input type="url" id="payhive_success_url" name="success_url" class="regular-text" />
                                    <p class="description"><?php esc_html_e('Optional: Redirect URL after successful payment.', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="payhive_cancel_url"><?php esc_html_e('Cancel URL', 'payhive-payments'); ?></label></th>
                                <td>
                                    <input type="url" id="payhive_cancel_url" name="cancel_url" class="regular-text" />
                                    <p class="description"><?php esc_html_e('Optional: Redirect URL if payment is cancelled.', 'payhive-payments'); ?></p>
                                </td>
                            </tr>
                        </table>
                        
                        <p class="submit">
                            <button type="submit" class="button button-primary">
                                <?php esc_html_e('Create Payment Link', 'payhive-payments'); ?>
                            </button>
                            <span id="payhive_create_loader" style="display:none;margin-left:10px;">
                                <span class="spinner" style="visibility:visible;float:none;"></span>
                            </span>
                        </p>
                        
                        <div id="payhive_create_result" class="payhive-result" style="display:none; margin-top:10px;"></div>
                    </form>
                </div>

                <!-- Payment Links List -->
                <div class="payhive-links-list-section" style="margin-top: 40px;">
                    <h2><?php esc_html_e('Your Payment Links', 'payhive-payments'); ?></h2>
                    <button type="button" id="payhive_refresh_links" class="button button-secondary">
                        <?php esc_html_e('Refresh List', 'payhive-payments'); ?>
                    </button>
                    <div id="payhive_links_list" style="margin-top: 20px;">
                        <p><?php esc_html_e('Loading payment links...', 'payhive-payments'); ?></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}

