<?php
/**
 * PAYHIIVE Payment Gateway for WooCommerce
 *
 * @package PayHive_Payments
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Check if WooCommerce is active
if (!class_exists('WC_Payment_Gateway')) {
    return;
}

/**
 * WC_Gateway_PayHive class.
 *
 * @extends WC_Payment_Gateway
 */
class WC_Gateway_PayHive extends WC_Payment_Gateway
{

    /**
     * Gateway ID.
     *
     * @var string
     */
    public $id = 'payhive';

    /**
     * Gateway title.
     *
     * @var string
     */
    public $method_title = 'PAYHIIVE Mobile Money';

    /**
     * Gateway description.
     *
     * @var string
     */
    public $method_description = 'Accept MTN & Airtel Mobile Money payments via PAYHIIVE';

    /**
     * Whether the gateway has fields.
     *
     * @var bool
     */
    public $has_fields = true;

    /**
     * Supported features.
     *
     * @var array
     */
    public $supports = array(
        'products',
    );

    /**
     * Constructor for the gateway.
     */
    public function __construct()
    {
        // Set basic gateway properties
        $this->id = 'payhive';
        $this->icon = ''; // Icon is handled by get_icon() method to control size
        $this->has_fields = true;
        $this->method_title = __('PAYHIIVE Mobile Money', 'payhive-payments');
        $this->method_description = __('Accept MTN & Airtel Mobile Money payments via PAYHIIVE API', 'payhive-payments');

        // Load the settings
        $this->init_form_fields();
        $this->init_settings();

        // Define user set variables
        $this->title = $this->get_option('title', 'PAYHIIVE Mobile Money');
        $this->description = $this->get_option('description', 'Pay with MTN Mobile Money or Airtel Money');
        $this->enabled = $this->get_option('enabled', 'no');
        $this->default_provider = $this->get_option('default_provider', 'mtn');

        // Save settings
        if (is_admin()) {
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        }

        // Enqueue checkout styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_checkout_styles'));
    }

    /**
     * Get the icon for the gateway.
     * No icon - only MTN and Airtel logos are shown in payment fields.
     *
     * @return string
     */
    public function get_icon()
    {
        // Return empty string - no PayHive logo, only MTN/Airtel logos in payment fields
        return '';
    }

    /**
     * Initialize Gateway Settings Form Fields.
     */
    public function init_form_fields()
    {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'payhive-payments'),
                'type' => 'checkbox',
                'label' => __('Enable PAYHIIVE Mobile Money', 'payhive-payments'),
                'default' => 'no',
            ),
            'title' => array(
                'title' => __('Title', 'payhive-payments'),
                'type' => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'payhive-payments'),
                'default' => __('PAYHIIVE Mobile Money', 'payhive-payments'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('Description', 'payhive-payments'),
                'type' => 'textarea',
                'description' => __('Payment method description that the customer will see on your checkout.', 'payhive-payments'),
                'default' => __('Pay securely with MTN Mobile Money or Airtel Money', 'payhive-payments'),
                'desc_tip' => true,
            ),
            'default_provider' => array(
                'title' => __('Default Provider', 'payhive-payments'),
                'type' => 'select',
                'description' => __('Default mobile money provider to pre-select on checkout.', 'payhive-payments'),
                'default' => 'mtn',
                'options' => array(
                    'mtn' => __('MTN Mobile Money', 'payhive-payments'),
                    'airtel' => __('Airtel Money', 'payhive-payments'),
                ),
                'desc_tip' => true,
            ),
            'api_settings_notice' => array(
                'title' => __('API Settings', 'payhive-payments'),
                'type' => 'title',
                'description' => __('PAYHIIVE API credentials are configured in <a href="' . admin_url('options-general.php?page=payhive-payments') . '">PAYHIIVE Settings</a>.', 'payhive-payments'),
            ),
        );
    }

    /**
     * Enqueue checkout styles and scripts.
     */
    public function enqueue_checkout_styles()
    {
        // Only load on checkout page and if gateway is enabled
        if (!is_checkout() || $this->enabled !== 'yes') {
            return;
        }

        // Enqueue styles
        wp_enqueue_style(
            'payhive-checkout-style',
            PAYHIVE_PLUGIN_URL . 'assets/css/payhive-checkout.css',
            array(),
            PAYHIVE_VERSION
        );

        // Enqueue scripts for interactive selection
        wp_enqueue_script(
            'payhive-checkout-script',
            PAYHIVE_PLUGIN_URL . 'assets/js/payhive-checkout.js',
            array('jquery'),
            PAYHIVE_VERSION,
            true
        );
    }

    /**
     * Output payment fields on checkout.
     */
    public function payment_fields()
    {
        if ($this->description) {
            echo wp_kses_post(wpautop(wptexturize($this->description)));
        }

        // Get default provider from settings
        $default_provider = $this->default_provider;

        // Get merchant name if display merchant name is enabled
        $display_merchant_name = get_option('payhive_display_merchant_name', false);
        $merchant_name = get_option('payhive_merchant_name', 'Okao Emmanuel');

        // Logo URLs - using plugin assets directory
        $mtn_logo_url = PAYHIVE_PLUGIN_URL . 'assets/img/Icon_MoMo-1-1024x812.png';
        $airtel_logo_url = PAYHIVE_PLUGIN_URL . 'assets/img/airtel-logo-freelogovectors.net_.png';
        ?>
        <fieldset id="payhive-payment-fields" class="payhive-payment-fields">
            <legend><?php esc_html_e('Select Mobile Money Provider', 'payhive-payments'); ?></legend>
            
            <div class="payhive-provider-options">
                <label class="payhive-provider-option <?php echo $default_provider === 'mtn' ? 'selected' : ''; ?>" for="payhive_provider_mtn">
                    <input 
                        type="radio" 
                        name="payhive_provider" 
                        id="payhive_provider_mtn" 
                        value="mtn" 
                        class="payhive-provider-radio" 
                        <?php checked($default_provider, 'mtn'); ?>
                        required 
                    />
                    <div class="payhive-provider-card">
                        <div class="payhive-provider-logo">
                            <img src="<?php echo esc_url($mtn_logo_url); ?>" alt="<?php esc_attr_e('MTN Mobile Money', 'payhive-payments'); ?>" />
                        </div>
                        <div class="payhive-provider-label">
                            <?php esc_html_e('MTN Mobile Money', 'payhive-payments'); ?>
                        </div>
                    </div>
                </label>

                <label class="payhive-provider-option <?php echo $default_provider === 'airtel' ? 'selected' : ''; ?>" for="payhive_provider_airtel">
                    <input 
                        type="radio" 
                        name="payhive_provider" 
                        id="payhive_provider_airtel" 
                        value="airtel" 
                        class="payhive-provider-radio" 
                        <?php checked($default_provider, 'airtel'); ?>
                        required 
                    />
                    <div class="payhive-provider-card">
                        <div class="payhive-provider-logo">
                            <img src="<?php echo esc_url($airtel_logo_url); ?>" alt="<?php esc_attr_e('Airtel Money', 'payhive-payments'); ?>" />
                        </div>
                        <div class="payhive-provider-label">
                            <?php esc_html_e('Airtel Money', 'payhive-payments'); ?>
                        </div>
                    </div>
                </label>
            </div>

            <?php if ($display_merchant_name && !empty($merchant_name)) : ?>
            <div class="payhive-merchant-name-display">
                <p class="payhive-merchant-name-text">
                    <strong><?php esc_html_e('Merchant:', 'payhive-payments'); ?></strong>
                    <span><?php echo esc_html($merchant_name); ?></span>
                </p>
            </div>
            <?php endif; ?>

            <p class="form-row form-row-wide payhive-phone-field">
                <label for="payhive_phone">
                    <?php esc_html_e('Phone Number', 'payhive-payments'); ?>
                    <span class="required">*</span>
                </label>
                <input 
                    type="tel" 
                    class="input-text" 
                    name="payhive_phone" 
                    id="payhive_phone" 
                    placeholder="<?php esc_attr_e('256XXXXXXXXX', 'payhive-payments'); ?>"
                    required 
                    autocomplete="tel" 
                />
                <small class="payhive-phone-hint">
                    <?php esc_html_e('Enter your mobile money phone number (e.g., 256712345678)', 'payhive-payments'); ?>
                </small>
            </p>
        </fieldset>
        <?php
    }

    /**
     * Validate payment fields on checkout.
     *
     * @return bool
     */
    public function validate_fields()
    {
        $provider = isset($_POST['payhive_provider']) ? sanitize_text_field($_POST['payhive_provider']) : '';
        $phone = isset($_POST['payhive_phone']) ? sanitize_text_field($_POST['payhive_phone']) : '';

        // Validate provider - must be either 'mtn' or 'airtel'
        if (empty($provider)) {
            wc_add_notice(__('Please select a mobile money provider (MTN or Airtel).', 'payhive-payments'), 'error');
            return false;
        }

        if (!in_array($provider, array('mtn', 'airtel'), true)) {
            wc_add_notice(__('Invalid mobile money provider selected. Please choose MTN or Airtel.', 'payhive-payments'), 'error');
            return false;
        }

        // Validate phone number
        if (empty($phone)) {
            wc_add_notice(__('Please enter your mobile money phone number.', 'payhive-payments'), 'error');
            return false;
        }

        // Basic phone number validation (should start with country code)
        $phone = preg_replace('/[^0-9]/', '', $phone);
        if (strlen($phone) < 9 || strlen($phone) > 15) {
            wc_add_notice(__('Please enter a valid phone number (9-15 digits).', 'payhive-payments'), 'error');
            return false;
        }

        return true;
    }

    /**
     * Process the payment and return the result.
     *
     * @param int $order_id Order ID.
     * @return array
     */
    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);

        if (!$order) {
            return array(
                'result' => 'fail',
                'redirect' => '',
            );
        }

        // Validate payment fields
        if (!$this->validate_fields()) {
            return array(
                'result' => 'fail',
                'redirect' => '',
            );
        }

        // Get payment data from POST and sanitize
        $provider = isset($_POST['payhive_provider']) ? sanitize_text_field($_POST['payhive_provider']) : '';
        $phone = isset($_POST['payhive_phone']) ? sanitize_text_field($_POST['payhive_phone']) : '';

        // Ensure provider is valid (must be 'mtn' or 'airtel')
        if (empty($provider) || !in_array($provider, array('mtn', 'airtel'), true)) {
            // Fallback to default if invalid
            $provider = $this->default_provider;
        }

        // Clean phone number (remove any non-numeric characters)
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // Ensure phone number starts with country code
        // If phone starts with 0, replace with 256 (Uganda country code)
        if (substr($phone, 0, 1) === '0') {
            $phone = '256' . substr($phone, 1);
        } elseif (substr($phone, 0, 3) !== '256') {
            // If it doesn't start with 256, add it
            $phone = '256' . $phone;
        }

        // Prepare payment data for PAYHIIVE API
        // Provider value must be exactly 'mtn' or 'airtel' (PAYHIIVE API will convert to MTN_MOMO_UGA or AIRTEL_OAPI_UGA)
        $payment_data = array(
            'amount' => $order->get_total(),
            'currency' => $order->get_currency(),
            'provider' => $provider, // 'mtn' or 'airtel' - PAYHIIVE_API will convert to API format
            'phone' => $phone,
            'description' => sprintf(
                /* translators: 1: Order number, 2: Site name */
                __('Order #%1$s on %2$s', 'payhive-payments'),
                $order->get_order_number(),
                get_bloginfo('name')
            ),
        );

        // Get PAYHIIVE API instance
        $payhive_api = new PayHive_API();
        
        // Process payment with PAYHIIVE
        $result = $payhive_api->process_mobile_payment($payment_data);

        // Check if payment was successful
        if (is_wp_error($result)) {
            // Log error
            $order->add_order_note(
                sprintf(
                    /* translators: 1: Error message */
                    __('PAYHIIVE payment failed: %s', 'payhive-payments'),
                    $result->get_error_message()
                )
            );

            wc_add_notice(
                sprintf(
                    /* translators: 1: Error message */
                    __('Payment error: %s', 'payhive-payments'),
                    $result->get_error_message()
                ),
                'error'
            );

            return array(
                'result' => 'fail',
                'redirect' => '',
            );
        }

        // Check if API response indicates success
        if (isset($result['success']) && $result['success'] === true) {
            // Payment request was successful
            // Mark order as on-hold (payment is pending confirmation from mobile money)
            $order->update_status('on-hold', __('PAYHIIVE payment initiated. Waiting for mobile money confirmation.', 'payhive-payments'));

            // Store payment metadata
            $order->update_meta_data('_payhive_provider', $provider);
            $order->update_meta_data('_payhive_phone', $phone);
            $order->update_meta_data('_payhive_transaction_id', isset($result['data']['transaction_id']) ? $result['data']['transaction_id'] : '');
            $order->update_meta_data('_payhive_deposit_id', isset($result['data']['deposit_id']) ? $result['data']['deposit_id'] : '');
            $order->save_meta_data();

            // Add order note
            $order->add_order_note(
                sprintf(
                    /* translators: 1: Provider, 2: Phone number, 3: Transaction ID */
                    __('PAYHIIVE payment initiated via %1$s to phone %2$s. Transaction ID: %3$s', 'payhive-payments'),
                    strtoupper($provider),
                    $phone,
                    isset($result['data']['transaction_id']) ? $result['data']['transaction_id'] : 'N/A'
                )
            );

            // Remove cart
            WC()->cart->empty_cart();

            // Return success
            return array(
                'result' => 'success',
                'redirect' => $this->get_return_url($order),
            );
        } else {
            // Payment failed
            $error_message = isset($result['message']) ? $result['message'] : __('Payment request failed. Please try again.', 'payhive-payments');
            
            $order->add_order_note(
                sprintf(
                    /* translators: 1: Error message */
                    __('PAYHIIVE payment failed: %s', 'payhive-payments'),
                    $error_message
                )
            );

            wc_add_notice($error_message, 'error');

            return array(
                'result' => 'fail',
                'redirect' => '',
            );
        }
    }
}




