<?php
/**
 * Plugin Name: PAYHIIVE Payments
 * Plugin URI: https://payhiive.com/
 * Description: Accept mobile money payments (MTN, Airtel) across East Africa via PAYHIIVE API with support for sandbox and live modes.
 * Version: 1.0.0
 * Author: PAYHIIVE TEAM
 * Author URI: https://payhiive.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: payhive-payments
 * Domain Path: /languages
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('PAYHIVE_VERSION', '1.0.0');
define('PAYHIVE_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('PAYHIVE_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PAYHIVE_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main PAYHIIVE Payments class.
 */
class PayHive_Payments
{

    /**
     * The single instance of the class.
     *
     * @var PayHive_Payments
     */
    protected static $_instance = null;

    /**
     * Main PayHive_Payments Instance.
     *
     * Ensures only one instance of PayHive_Payments is loaded or can be loaded.
     *
     * @return PayHive_Payments - Main instance.
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * PayHive_Payments Constructor.
     */
    public function __construct()
    {
        $this->includes();
        $this->init_hooks();
    }

    /**
     * Include required core files.
     */
    public function includes()
    {
        require_once PAYHIVE_PLUGIN_DIR . 'includes/class-payhive-admin.php';
        require_once PAYHIVE_PLUGIN_DIR . 'includes/class-payhive-api.php';
        require_once PAYHIVE_PLUGIN_DIR . 'includes/class-payhive-shortcode.php';
        require_once PAYHIVE_PLUGIN_DIR . 'includes/class-payhive-payment-links.php';
    }

    /**
     * Include WooCommerce gateway class when WooCommerce is loaded.
     */
    public function load_woocommerce_gateway()
    {
        // Load WooCommerce gateway class only if WooCommerce is active
        if ($this->is_woocommerce_active() && !class_exists('WC_Gateway_PayHive')) {
            require_once PAYHIVE_PLUGIN_DIR . 'includes/class-wc-gateway-payhive.php';
        }
    }

    /**
     * Hook into actions and filters.
     */
    private function init_hooks()
    {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        add_action('plugins_loaded', array($this, 'init'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

        // Load WooCommerce gateway when WooCommerce is fully loaded
        add_action('woocommerce_loaded', array($this, 'load_woocommerce_gateway'));

        // Register WooCommerce payment gateway (filter checks if class exists before adding)
        add_filter('woocommerce_payment_gateways', array($this, 'add_woocommerce_gateway'));

        // Show admin notice if WooCommerce is not active
        add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
    }

    /**
     * Initialize the plugin.
     */
    public function init()
    {
        // Initialize admin settings
        if (is_admin()) {
            PayHive_Admin::instance();
        }

        // Initialize shortcode
        PayHive_Shortcode::instance();

        // Initialize payment links manager
        if (is_admin()) {
            PayHive_Payment_Links::instance();
        }

        do_action('payhive_payments_loaded');
    }

    /**
     * Enqueue admin scripts and styles.
     */
    public function admin_scripts($hook)
    {
        // Only load on our settings page
        if ('settings_page_payhive-payments' !== $hook) {
            return;
        }

        wp_enqueue_style(
            'payhive-admin-style',
            PAYHIVE_PLUGIN_URL . 'assets/css/admin-style.css',
            array(),
            PAYHIVE_VERSION
        );

        wp_enqueue_script(
            'payhive-admin-script',
            PAYHIVE_PLUGIN_URL . 'assets/js/admin-script.js',
            array('jquery'),
            PAYHIVE_VERSION,
            true
        );

        // Localize script to pass PHP variables to JavaScript
        wp_localize_script(
            'payhive-admin-script',
            'payhiveAdmin',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('payhive_test_nonce'),
            )
        );
    }

    /**
     * Plugin activation callback.
     */
    public function activate()
    {
        // Set default options on activation
        $defaults = array(
            'payhive_api_key' => '',
            'payhive_secret_key' => '',
            'payhive_mode' => 'sandbox',
            'payhive_default_fee' => '0',
            'payhive_display_merchant_name' => false,
            'payhive_merchant_name' => 'Okao Emmanuel',
        );

        foreach ($defaults as $key => $value) {
            if (false === get_option($key)) {
                add_option($key, $value);
            }
        }

        // Log activation
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('PAYHIIVE Payments plugin activated');
        }
    }

    /**
     * Plugin deactivation callback.
     */
    public function deactivate()
    {
        // Log deactivation
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('PAYHIIVE Payments plugin deactivated');
        }
    }

    /**
     * Check if WooCommerce is active.
     *
     * @return bool
     */
    public function is_woocommerce_active()
    {
        return class_exists('WooCommerce');
    }

    /**
     * Add PayHive gateway to WooCommerce payment gateways.
     *
     * @param array $gateways Existing payment gateways.
     * @return array
     */
    public function add_woocommerce_gateway($gateways)
    {
        // Only add gateway if WooCommerce is active and gateway class exists
        if ($this->is_woocommerce_active() && class_exists('WC_Gateway_PayHive')) {
            $gateways[] = 'WC_Gateway_PayHive';
        }

        return $gateways;
    }

    /**
     * Show admin notice if WooCommerce is not active.
     */
    public function woocommerce_missing_notice()
    {
        // Only show notice in admin area
        if (!is_admin()) {
            return;
        }

        // Only show notice if WooCommerce is not active
        if ($this->is_woocommerce_active()) {
            return;
        }

        // Only show on plugin pages or WooCommerce pages
        $screen = get_current_screen();
        if (!$screen || (
            strpos($screen->id, 'payhive') === false &&
            strpos($screen->id, 'woocommerce') === false &&
            strpos($screen->id, 'plugins') === false
        )) {
            return;
        }

        ?>
        <div class="notice notice-info is-dismissible">
            <p>
                <strong><?php esc_html_e('PAYHIIVE Payments', 'payhive-payments'); ?></strong>:
                <?php
                printf(
                    /* translators: 1: WooCommerce plugin name */
                    esc_html__('WooCommerce is required for the PAYHIIVE payment gateway. %1$s', 'payhive-payments'),
                    '<a href="' . esc_url(admin_url('plugin-install.php?s=woocommerce&tab=search&type=term')) . '">' . esc_html__('Install WooCommerce', 'payhive-payments') . '</a>'
                );
                ?>
            </p>
        </div>
        <?php
    }
}

/**
 * Returns the main instance of PayHive_Payments.
 *
 * @return PayHive_Payments
 */
function payhive_payments()
{
    return PayHive_Payments::instance();
}

// Initialize the plugin
payhive_payments();
